<?php
// Include the database connection
include '../dbconn.php';

// Check if the request method is POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Retrieve the input from the form data
    $input = $_POST['input'] ?? null;

    // Initialize the response
    $response = [
        "status" => "error",
        "message" => "Failed to send OTP. Please try again later."
    ];

    // Validate the input
    $isMobile = $input && preg_match('/^\d{10}$/', $input); // 10-digit mobile number
    $isEmail = $input && filter_var($input, FILTER_VALIDATE_EMAIL); // Valid email

    if ($isMobile) {
        $identifier = "mobile";
        $mobile = $input;
        $email = null;
    } elseif ($isEmail) {
        $identifier = "email";
        $email = $input;
        $mobile = null;
    } else {
        // Invalid input format
        $response['message'] = "Invalid input format. Please provide a valid 10-digit mobile number or email address.";
        echo json_encode($response);
        exit;
    }

    // Check if the mobile number or email is registered
    $query = $identifier === "mobile"
        ? "SELECT id FROM users WHERE mobile = ?"
        : "SELECT id FROM users WHERE email = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("s", $input);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        // Input not registered
        $response['message'] = $identifier === "mobile"
            ? "Mobile number not registered."
            : "Email not registered.";
        echo json_encode($response);
        exit;
    }

    // Generate a 4-digit OTP (for testing, use a fixed OTP like '1234')
    $otp = rand(1000, 9999);

    // Check if an OTP already exists for the input in the OTP table
    $otpQuery = $identifier === "mobile"
        ? "SELECT created_at FROM otp_store WHERE mobile = ?"
        : "SELECT created_at FROM otp_store WHERE email = ?";
    $otpStmt = $conn->prepare($otpQuery);
    $otpStmt->bind_param("s", $input);
    $otpStmt->execute();
    $otpResult = $otpStmt->get_result();

    if ($otpResult->num_rows === 1) {
        // Update the existing OTP
        $updateQuery = $identifier === "mobile"
            ? "UPDATE otp_store SET otp = ?, created_at = NOW() WHERE mobile = ?"
            : "UPDATE otp_store SET otp = ?, created_at = NOW() WHERE email = ?";
        $updateStmt = $conn->prepare($updateQuery);
        $updateStmt->bind_param("is", $otp, $input);
        $updateSuccess = $updateStmt->execute();
    } else {
        // Insert a new OTP record
        $insertQuery = $identifier === "mobile"
            ? "INSERT INTO otp_store (mobile, otp, created_at) VALUES (?, ?, NOW())"
            : "INSERT INTO otp_store (email, otp, created_at) VALUES (?, ?, NOW())";
        $insertStmt = $conn->prepare($insertQuery);
        $insertStmt->bind_param("si", $input, $otp);
        $insertSuccess = $insertStmt->execute();
    }

    // Check if the OTP was successfully stored
    if (isset($updateSuccess) && $updateSuccess || isset($insertSuccess) && $insertSuccess) {
        // Send OTP based on the identifier
        if ($identifier === "email") {
            // Simulate email sending
            $subject = "Your OTP Code";
            $output = "Your OTP code is: $otp. Please use this to reset your password.";
            $emailSent = forgotpassword($email, $output, $subject);

            if ($emailSent) {
                $response = [
                    "status" => "success",
                    "message" => "OTP sent successfully to the provided email.",
                ];
            } else {
                $response['message'] = "Failed to send OTP via email. Please try again.";
            }
        } elseif ($identifier === "mobile") {
            // Simulate SMS sending (add your logic here)
            $response = [
                "status" => "success",
                "message" => "OTP sent successfully to the provided mobile number.",
                "otp" => $otp // Include OTP in response for testing (remove in production)
            ];
        }
    }

    // Output the response
    echo json_encode($response);
    $stmt->close();
    $conn->close();
} else {
    // Return error if the request method is not POST
    echo json_encode([
        "status" => "error",
        "message" => "Invalid request method. Use POST."
    ]);
}
?>
