<?php
// Disable error display for production
error_reporting(0);
ini_set('display_errors', 0);

require('../dbconn.php');
require('../fpdf.php');

// Start output buffering to prevent any output before PDF
ob_start();

if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['order_id'])) {
    $order_id = intval($_GET['order_id']);

    // Fetch order details, including shipping charges
    $query = "SELECT orders.id, orders.user_id, orders.total_amount, COALESCE(orders.discount_amount, 0) as discount_amount, 
                     COALESCE(orders.shipping_charges, 0) as shipping_charges, orders.payment_status, orders.payment_method, orders.created_at,
                     shipping_addresses.first_name, shipping_addresses.last_name, 
                     COALESCE(shipping_addresses.phone_number, 'N/A') as phone_number,
                     shipping_addresses.city, shipping_addresses.state, shipping_addresses.country
              FROM orders
              JOIN shipping_addresses ON orders.shipping_address_id = shipping_addresses.id
              WHERE orders.id = ?";

    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $order_id);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows > 0) {
        $order = $result->fetch_assoc();

        // Fetch order items
        $items_query = "SELECT products.name, 
                              order_items.quantity, 
                              COALESCE(order_items.unit_price, 0) as unit_price, 
                              COALESCE(order_items.total_price, 0) as total_price
                       FROM order_items
                       JOIN products ON order_items.product_id = products.id
                       WHERE order_items.order_id = ?";
        $stmt = $conn->prepare($items_query);
        $stmt->bind_param("i", $order_id);
        $stmt->execute();
        $items_result = $stmt->get_result();

        // Generate PDF
        class PDF extends FPDF {
            function Header() {
                // Add logo on the left side
                $this->Image('logo.png', 10, 10, 40);
                
                // Move to position after logo for title
                $this->SetY(15); // Adjust Y position to vertically center with logo
                
                // Title - moved to right of logo
                $this->SetFont('Arial', 'B', 16);
                $this->Cell(0, 10, 'Order Invoice', 0, 1, 'C');
                
                // Add a line separator
                $this->SetY(30);
                $this->Line(10, $this->GetY(), 200, $this->GetY());
                $this->Ln(5);
            }
            function Footer() {
                $this->SetY(-15);
                $this->SetFont('Arial', 'I', 10);
                $this->Cell(0, 10, 'Thank you for your order!', 0, 0, 'C');
            }
        }

        try {
            // Clear any output buffers
            ob_clean();
            
            $pdf = new PDF();
            $pdf->AddPage();
            $pdf->SetFont('Arial', '', 12);

            // Order Info
            $pdf->Cell(0, 10, "Order ID: {$order['id']}", 0, 1);
            $pdf->Cell(0, 10, "Customer: {$order['first_name']} {$order['last_name']}", 0, 1);
            $pdf->Cell(0, 10, "Phone: {$order['phone_number']}", 0, 1);
            $pdf->Cell(0, 10, "Shipping Address: {$order['city']}, {$order['state']}, {$order['country']}", 0, 1);
            $pdf->Ln(5);

            // Table Header
            $pdf->SetFont('Arial', 'B', 12);
            $pdf->Cell(90, 10, 'Product', 1);
            $pdf->Cell(30, 10, 'Quantity', 1);
            $pdf->Cell(30, 10, 'Unit Price', 1);
            $pdf->Cell(40, 10, 'Total Price', 1);
            $pdf->Ln();

            // Table Content
            $pdf->SetFont('Arial', '', 12);
            while ($item = $items_result->fetch_assoc()) {
                $pdf->Cell(90, 10, $item['name'], 1);
                $pdf->Cell(30, 10, $item['quantity'], 1);
                $pdf->Cell(30, 10, number_format($item['unit_price'], 2), 1);
                $pdf->Cell(40, 10, number_format($item['total_price'], 2), 1);
                $pdf->Ln();
            }

            // Summary
            $pdf->Ln(5);
            $pdf->Cell(0, 10, "Shipping Charges: " . number_format($order['shipping_charges'], 2), 0, 1);
            $pdf->Cell(0, 10, "Total Amount: " . number_format($order['total_amount'], 2), 0, 1);
            $pdf->Cell(0, 10, "Discount: " . number_format($order['discount_amount'], 2), 0, 1);
            $pdf->Cell(0, 10, "Final Total: " . number_format($order['total_amount'] + $order['shipping_charges'] - $order['discount_amount'], 2), 0, 1);
            $pdf->Cell(0, 10, "Payment Method: {$order['payment_method']}", 0, 1);

            // Output PDF
            $pdf->Output('D', "invoice_{$order_id}.pdf");
            exit();
        } catch (Exception $e) {
            // Log error for debugging
            error_log("PDF Generation Error: " . $e->getMessage());
            header('Content-Type: application/json');
            echo json_encode(["success" => false, "message" => "Error generating PDF"]);
            exit();
        }
    } else {
        header('Content-Type: application/json');
        echo json_encode(["success" => false, "message" => "Order not found"]);
    }
} else {
    header('Content-Type: application/json');
    echo json_encode(["success" => false, "message" => "Invalid request"]);
}
?>
