<?php
include '../../dbconn.php'; // Database connection

header("Content-Type: application/json");

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    echo json_encode(["success" => false, "message" => "Invalid request method"]);
    exit;
}

// Validate promocode ID
if (!isset($_GET['id']) || empty($_GET['id'])) {
    echo json_encode(["success" => false, "message" => "Promocode ID is required"]);
    exit;
}

$promocode_id = intval($_GET['id']);
$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : null;
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : null;
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$limit = isset($_GET['limit']) ? intval($_GET['limit']) : 10;
$offset = ($page - 1) * $limit;

// Fetch promocode details
$promocode_query = "
    SELECT id, code, times_used, 
           (SELECT SUM(discount_amount) FROM promocode_usage WHERE promocode_id = ?) AS total_discount_amount 
    FROM promocodes WHERE id = ?
";
$stmt = $conn->prepare($promocode_query);
$stmt->bind_param("ii", $promocode_id, $promocode_id);
$stmt->execute();
$promocode_result = $stmt->get_result();
$promocode = $promocode_result->fetch_assoc();

if (!$promocode) {
    echo json_encode(["success" => false, "message" => "Promocode not found"]);
    exit;
}

// Fetch usage history
$usage_query = "
    SELECT pu.id, pu.order_id, pu.user_id, u.name AS user_name, o.total_amount AS order_amount, 
           pu.discount_amount, pu.created_at
    FROM promocode_usage pu
    JOIN users u ON pu.user_id = u.id
    JOIN orders o ON pu.order_id = o.id
    WHERE pu.promocode_id = ?
";

$params = [$promocode_id];
$types = "i";

if ($date_from) {
    $usage_query .= " AND pu.created_at >= ?";
    $params[] = $date_from;
    $types .= "s";
}
if ($date_to) {
    $usage_query .= " AND pu.created_at <= ?";
    $params[] = $date_to;
    $types .= "s";
}

$usage_query .= " ORDER BY pu.created_at DESC LIMIT ? OFFSET ?";
$params[] = $limit;
$params[] = $offset;
$types .= "ii";

$stmt = $conn->prepare($usage_query);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$usage_result = $stmt->get_result();

$usage_history = [];
while ($row = $usage_result->fetch_assoc()) {
    $usage_history[] = $row;
}

// Fetch total count for pagination
$count_query = "SELECT COUNT(*) AS total FROM promocode_usage WHERE promocode_id = ?";
$stmt = $conn->prepare($count_query);
$stmt->bind_param("i", $promocode_id);
$stmt->execute();
$count_result = $stmt->get_result();
$total_count = $count_result->fetch_assoc()['total'];

// Fetch statistics
$stats_query = "
    SELECT AVG(discount_amount) AS average_discount,
           AVG(o.total_amount) AS average_order_value
    FROM promocode_usage pu
    JOIN orders o ON pu.order_id = o.id
    WHERE pu.promocode_id = ?
";
$stmt = $conn->prepare($stats_query);
$stmt->bind_param("i", $promocode_id);
$stmt->execute();
$stats_result = $stmt->get_result();
$statistics = $stats_result->fetch_assoc();

// Fetch usage by date
$date_usage_query = "
    SELECT DATE(pu.created_at) AS date, COUNT(*) AS count, SUM(pu.discount_amount) AS total_discount
    FROM promocode_usage pu
    WHERE pu.promocode_id = ?
    GROUP BY DATE(pu.created_at)
    ORDER BY date DESC
";
$stmt = $conn->prepare($date_usage_query);
$stmt->bind_param("i", $promocode_id);
$stmt->execute();
$date_usage_result = $stmt->get_result();

$usage_by_date = [];
while ($row = $date_usage_result->fetch_assoc()) {
    $usage_by_date[] = $row;
}

// Response
$response = [
    "success" => true,
    "data" => [
        "promocode" => $promocode,
        "usage_history" => $usage_history,
        "pagination" => [
            "total" => $total_count,
            "page" => $page,
            "limit" => $limit
        ],
        "statistics" => [
            "average_discount" => $statistics["average_discount"] ?? 0,
            "average_order_value" => $statistics["average_order_value"] ?? 0,
            "usage_by_date" => $usage_by_date
        ]
    ]
];

echo json_encode($response);
$conn->close();
?>
