<?php
header("Content-Type: application/json");
require '../dbconn.php'; // Include database connection
require('../fpdf.php'); // Include FPDF library

$response = ["success" => false];

if ($_SERVER["REQUEST_METHOD"] === "POST") {
    $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
    $updated_amount = isset($_POST['updated_amount']) ? floatval($_POST['updated_amount']) : 0;
    
    if ($id > 0 && $updated_amount >= 0) {
        // Initialize statements to null to prevent undefined variable errors
        $quote_stmt = null;
        $user_stmt = null;
        $cart_stmt = null;
        $insert_stmt = null;

        try {
            // First, fetch the existing quote details
            $quote_stmt = $conn->prepare("SELECT user_id, quote_pdf, total_price FROM quotes WHERE id = ?");
            if (!$quote_stmt) {
                throw new Exception("Failed to prepare quote statement: " . $conn->error);
            }
            $quote_stmt->bind_param("i", $id);
            if (!$quote_stmt->execute()) {
                throw new Exception("Failed to execute quote query: " . $quote_stmt->error);
            }
            $quote_result = $quote_stmt->get_result();
            
            if ($quote_result->num_rows === 0) {
                throw new Exception("Original quote not found");
            }
            
            $quote_details = $quote_result->fetch_assoc();
            
            // Fetch user details
            $user_stmt = $conn->prepare("SELECT name, mobile, email FROM users WHERE id = ?");
            if (!$user_stmt) {
                throw new Exception("Failed to prepare user statement: " . $conn->error);
            }
            $user_stmt->bind_param("i", $quote_details['user_id']);
            if (!$user_stmt->execute()) {
                throw new Exception("Failed to execute user query: " . $user_stmt->error);
            }
            $user_result = $user_stmt->get_result();
            
            if ($user_result->num_rows === 0) {
                throw new Exception("User not found");
            }
            
            $user_details = $user_result->fetch_assoc();
            
            // Fetch cart items for the quote
            $cart_stmt = $conn->prepare("SELECT 
                products.name AS Name,
                products.sku, 
                temp_cart.quantity, 
                temp_cart.unit_price
            FROM temp_cart 
            INNER JOIN products ON temp_cart.product_id = products.id 
            WHERE temp_cart.user_id = ? AND temp_cart.is_checked_out = 0");
            if (!$cart_stmt) {
                throw new Exception("Failed to prepare cart statement: " . $conn->error);
            }
            $cart_stmt->bind_param("i", $quote_details['user_id']);
            if (!$cart_stmt->execute()) {
                throw new Exception("Failed to execute cart query: " . $cart_stmt->error);
            }
            $cart_result = $cart_stmt->get_result();
            
            // Prepare cart items and calculate subtotals
            $cart_items = [];
            $subtotal = 0;
            while ($row = $cart_result->fetch_assoc()) {
                $row['subtotal'] = $row['quantity'] * $row['unit_price'];
                $subtotal += $row['subtotal'];
                $cart_items[] = $row;
            }
            
            // GST Calculations
            $gst_rate = 18; // 18% GST
            $sgst = ($subtotal * 9) / 100;
            $cgst = ($subtotal * 9) / 100;
            $total_amount = $subtotal + $sgst + $cgst;
            $total_price = $updated_amount; // Use the updated amount

            // Generate PDF using FPDF
            class PDF extends FPDF {
                function Header() {
                    // Company logo at the top left
                    $this->Image('logo.png', 20, 10, 30); // Adjust the path and size

                    // Title at center
                    $this->SetFont('Arial', 'B', 14);
                    $this->SetY(30);
                    $this->Cell(0, 10, 'Updated Quotation', 0, 1, 'C');
                }

                function Footer() {
                    $this->SetY(-15);
                    $this->SetFont('Arial', 'I', 10);
                    $this->Cell(0, 10, 'Thank you for shopping with us!', 0, 0, 'C');
                }
            }

            $pdf = new PDF();
            $pdf->AddPage();
            $pdf->SetFont('Arial', 'B', 12);

            // Adjust Y position to center the table
            $pdf->SetY(42);
            $pdf->SetX(20);

            // Table Headers
            $pdf->Cell(60, 10, 'Product Name', 1);
            $pdf->Cell(30, 10, 'SKU', 1);
            $pdf->Cell(20, 10, 'Qty', 1);
            $pdf->Cell(30, 10, 'Unit Price', 1);
            $pdf->Cell(30, 10, 'Subtotal', 1);
            $pdf->Ln();

            // Table Data
            $pdf->SetFont('Arial', '', 12);
            foreach ($cart_items as $item) {
                $pdf->SetX(20);
                $pdf->Cell(60, 10, utf8_decode($item['Name']), 1);
                $pdf->Cell(30, 10, $item['sku'], 1);
                $pdf->Cell(20, 10, $item['quantity'], 1);
                $pdf->Cell(30, 10, number_format($item['unit_price'], 2), 1);
                $pdf->Cell(30, 10, number_format($item['subtotal'], 2), 1);
                $pdf->Ln();
            }

            // Subtotal Row
            $pdf->SetFont('Arial', 'B', 12);
            $pdf->SetX(20);
            $pdf->Cell(140, 10, 'Subtotal:', 1, 0, 'R');
            $pdf->Cell(30, 10, number_format($subtotal, 2), 1, 1);

            // GST Breakdown
            $pdf->SetX(20);
            $pdf->Cell(140, 10, 'SGST (9%):', 1, 0, 'R');
            $pdf->Cell(30, 10, number_format($sgst, 2), 1, 1);
            
            $pdf->SetX(20);
            $pdf->Cell(140, 10, 'CGST (9%):', 1, 0, 'R');
            $pdf->Cell(30, 10, number_format($cgst, 2), 1, 1);

            // Total Price Row with Updated Amount
            $pdf->SetFont('Arial', '', 12);
            $pdf->SetX(20);
            $pdf->Cell(140, 12, 'Total Price (Updated):', 1, 0, 'R');
            $pdf->Cell(30, 12, 'RS. ' . number_format($total_price, 2), 1, 1);

            // Save PDF
            $quote_file = 'quotes/updated_invoice_' . $quote_details['user_id'] . '_' . time() . '.pdf';
            $pdf->Output('../' . $quote_file, 'F'); // Save to file

            // Insert new quote record with updated amount
            $insert_query = "INSERT INTO quotes (user_id, quote_pdf, total_price, updated_amount) VALUES (?, ?, ?, ?)";
            $insert_stmt = $conn->prepare($insert_query);
            if (!$insert_stmt) {
                throw new Exception("Failed to prepare insert statement: " . $conn->error);
            }
            $insert_stmt->bind_param("isdd", 
                $quote_details['user_id'], 
                $quote_file, 
                $total_amount, 
                $total_price
            );
            
            if (!$insert_stmt->execute()) {
                throw new Exception("Failed to insert updated quote: " . $insert_stmt->error);
            }
            
            // Delete the quote from quotes table (NEW ADDITION)
                $delete_stmt = $conn->prepare("DELETE FROM quotes WHERE id = ?");
                $delete_stmt->bind_param("i", $id);
                
                if (!$delete_stmt->execute()) {
                    throw new Exception("Failed to delete quote");
                }
                $delete_stmt->close();

            $response["success"] = true;
            $response["message"] = "Quote updated successfully";
            $response["quote_url"] = 'https://sandbox.boltkart.in/API/' . $quote_file;
            $response["user_details"] = [
                'name' => $user_details['name'],
                'phone' => $user_details['mobile'],
                'email' => $user_details['email']
            ];

        } catch (Exception $e) {
            $response["message"] = $e->getMessage();
        } finally {
            // Close all prepared statements if they exist
            if ($quote_stmt !== null) $quote_stmt->close();
            if ($user_stmt !== null) $user_stmt->close();
            if ($cart_stmt !== null) $cart_stmt->close();
            if ($insert_stmt !== null) $insert_stmt->close();
        }
    } else {
        $response["message"] = "Invalid input";
    }
} else {
    $response["message"] = "Invalid request method";
}

echo json_encode($response);
$conn->close();
?>