<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);

header("Content-Type: application/json");
require '../dbconn.php';

/* =======================
   PAGINATION
======================= */
$page  = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$limit = isset($_GET['limit']) ? max(1, intval($_GET['limit'])) : 20;

$halfLimit = floor($limit / 2);
$offset = ($page - 1) * $halfLimit;

/* =======================
   VARIANT JSON
======================= */
$variantJson = "
COALESCE(
    JSON_ARRAYAGG(
        JSON_OBJECT(
            'product_variant_id', pv.product_variant_id,
            'product_ref_id', pv.product_ref_id,
            'sku', pv.sku,
            'price', pv.price,
            'stock_quantity', pv.stock_quantity,
            'cut_and_clean_price', pv.cut_and_clean_price,
            'image_url',
                CASE
                    WHEN pv.image_url IS NULL OR pv.image_url = ''
                    THEN p.image_url
                    ELSE pv.image_url
                END,
            'specifications', pv.specifications,
            'is_default', pv.is_default,
            'in_stock', pv.in_stock,
            'name', pv.name,
            'description', pv.description,
            'created_at', pv.created_at,
            'updated_at', pv.updated_at
        )
    ),
    JSON_ARRAY()
) AS variants
";

/* =======================
   LATEST PRODUCTS
======================= */
$queryLatest = "
SELECT
    p.product_id,
    p.name,
    p.image_url,
    p.is_best_seller,
    p.is_latest,
    p.active,
    p.created_at,

    EXISTS (
        SELECT 1
        FROM product_variants pvx
        WHERE pvx.product_ref_id = p.product_id
    ) AS has_products,

    $variantJson

FROM products p
LEFT JOIN product_variants pv
    ON pv.product_ref_id = p.product_id

WHERE p.is_latest = 1
AND p.active = 1

GROUP BY p.product_id
ORDER BY p.created_at DESC
LIMIT ? OFFSET ?
";

$stmt1 = $conn->prepare($queryLatest);
$stmt1->bind_param("ii", $halfLimit, $offset);
$stmt1->execute();
$res1 = $stmt1->get_result();

$latestProducts = [];
while ($row = $res1->fetch_assoc()) {
    $row['variants'] = json_decode($row['variants'], true) ?: [];
    $row['has_products'] = (bool)$row['has_products'];
    $latestProducts[] = $row;
}

/* =======================
   BEST SELLER PRODUCTS
======================= */
$queryBest = "
SELECT
    p.product_id,
    p.name,
    p.image_url,
    p.is_best_seller,
    p.is_latest,
    p.active,
    p.created_at,

    EXISTS (
        SELECT 1
        FROM product_variants pvx
        WHERE pvx.product_ref_id = p.product_id
    ) AS has_products,

    $variantJson

FROM products p
LEFT JOIN product_variants pv
    ON pv.product_ref_id = p.product_id

WHERE p.is_best_seller = 1
AND p.active = 1

GROUP BY p.product_id
ORDER BY p.created_at DESC
LIMIT ? OFFSET ?
";

$stmt2 = $conn->prepare($queryBest);
$stmt2->bind_param("ii", $halfLimit, $offset);
$stmt2->execute();
$res2 = $stmt2->get_result();

$bestProducts = [];
while ($row = $res2->fetch_assoc()) {
    $row['variants'] = json_decode($row['variants'], true) ?: [];
    $row['has_products'] = (bool)$row['has_products'];
    $bestProducts[] = $row;
}

/* =======================
   COUNTS
======================= */
$countLatest = $conn->query("
    SELECT COUNT(*) AS total 
    FROM products 
    WHERE is_latest = 1 AND active = 1
")->fetch_assoc()['total'];

$countBest = $conn->query("
    SELECT COUNT(*) AS total 
    FROM products 
    WHERE is_best_seller = 1 AND active = 1
")->fetch_assoc()['total'];

$total = $countLatest + $countBest;

/* =======================
   RESPONSE
======================= */
echo json_encode([
    "success" => true,
    "page" => $page,
    "limit" => $limit,
    "each_limit" => $halfLimit,
    "total_latest" => $countLatest,
    "total_best_seller" => $countBest,
    "total_items" => $total,
    "total_pages" => ceil($total / $limit),
    "latest_products" => $latestProducts,
    "best_seller_products" => $bestProducts
]);

$stmt1->close();
$stmt2->close();
$conn->close();
